﻿using System;
using System.Windows.Forms;
using DataWindowPlugInEx.View;
using Tektronix.LogicAnalyzer.Common;
using Tektronix.LogicAnalyzer.PlugIn;
using Tektronix.LogicAnalyzer.TpiNet;

namespace DataWindowPlugInEx
{
    /// <summary>
    /// This is the object that will implement IDataWindowPlugIn.    
    /// </summary>
    /// <remarks>
    /// 1. Project should be targeted at .NET 2.0 or earlier (until support for 4.0 is added).
    /// 2. Any data window plug-in MUST implement IDataWindowPlugIn.
    /// 3. It is also recommended to implement IDataSourceChangeNotifier, so the system window
    ///    knows when to re-paint the arrows denoting which data sources the window includes.
    /// 4. The PlugInIdentity Attribute used below defines the text used in the New Data 
    ///    Window dialog to identify this kind of data window before an instance of it is 
    ///    created.    
    /// </remarks>
    [PlugInIdentity("DataWindowPlugInEx")]
    public class DataWindowPlugInEx : IDataWindowPlugIn, IDataSourceChangeNotifier 
    {
        private DataWindow DataWindowForm { get; set; }
        private SystemWindowControl DataWindowSystemControl { get; set; }
        public ITlaPlugInSupport Support { get; set; }

        #region Method(s)

        private void RaiseDataSourceSetChanged()
        {
            if (DataSourceSetChanged != null)
            {
                DataSourceSetChanged(this, EventArgs.Empty);
            }
        }

        private void RaiseUserNameChanged()
        {
            if (UserNameChanged != null)
            {
                UserNameChanged(this, EventArgs.Empty);
            }
        }

        private void RaiseValidityChanged()
        {
            if (ValidityChanged != null)
            {
                ValidityChanged(this, EventArgs.Empty);
            }
        }

        #endregion

        #region IDataWindowPlugIn Member(s)

        private bool _isValid = true;
        public bool IsValid
        {
            get { return _isValid; }
            set
            {
                if (_isValid != value)
                {
                    _isValid = value;
                    RaiseValidityChanged();
                }
            }
        }

        private string _userName = "DwPlugInEx 1";
        public string UserName
        {
            get { return _userName; }
            set
            {
                if (_userName != value)
                {
                    _userName = value;
                    RaiseUserNameChanged();
                }
            }
        }

        public bool IsGarbage { get; set; }

        public Form PrimaryForm
        {
            get { return DataWindowForm; }
        }

        public UserControl SystemWindowControl
        {
            get { return DataWindowSystemControl; }
        }

        public bool Initialize(ITlaPlugInSupport support, bool userInit)
        {
            Support = support;
            if (Support != null)
            {
                DataWindowForm = new DataWindow(this);
                DataWindowSystemControl = new SystemWindowControl(this);
                
                // Move the new data window into the main window of the application.
                // If this is not done, then the data window form will not show up inside
                // the application.
                Support.TopLevelFormToChild(DataWindowForm);

                // Listen for any change in the selected data source in the DataWindow.
                DataWindowForm.DataSourceChanged += DataWindowDataSourceChanged;

                // Listen for disposal (close) of the data window.
                DataWindowForm.Disposed += DataWindowFormDisposed;

                // Initialize the form.
                DataWindowForm.Initialize();
            }
            return true;
        }

        public IDataSource[] GetDataSources()
        {
            // Any IDataSource that is being viewed by this IDataWindowPlugIn should be returned here.
            // For this example, only one IDataSource is viewed at any given time.
            if (DataWindowForm != null && DataWindowForm.DataSource != null)
            {
                return new IDataSource[] { DataWindowForm.DataSource };
            }
            return null;
        }

        public void Dispose()
        {
            if (!IsGarbage)
            {
                // Unregister from any events, dispose, and clear out references to any object
                // created by the plug-in.
                if (DataWindowForm != null)
                {
                    DataWindowForm.DataSourceChanged -= DataWindowDataSourceChanged;
                    DataWindowForm.Disposed -= DataWindowFormDisposed;
                    DataWindowForm.Dispose();
                    DataWindowForm = null;
                }
                if (DataWindowSystemControl != null)
                {
                    DataWindowSystemControl.Dispose();
                    DataWindowSystemControl = null;
                }

                // Once disposed, set the "IsGarbage" flag to true.
                IsGarbage = true;

                // Clear the "IsValid" flag.  This also fires off the ValidityChanged event.
                IsValid = false;
            }
        }

        public event EventHandler ValidityChanged;
        public event EventHandler UserNameChanged;

        #endregion

        #region IDataSourceChangeNotifier

        public event EventHandler DataSourceSetChanged;

        #endregion

        #region Event Handler(s)

        private void DataWindowDataSourceChanged(object sender, EventArgs eventArgs)
        {
            RaiseDataSourceSetChanged();
        }

        private void DataWindowFormDisposed(object sender, EventArgs e)
        {
            Dispose();
        }

        #endregion
    }
}
